<?php
/**
 * Copyright © 2016 Magentix. All rights reserved.
 *
 * NOTICE OF LICENSE
 * This source file is subject to commercial licence, do not copy or distribute without authorization
 */

class LaPoste_Colissimo_Helper_Data extends Mage_Core_Helper_Abstract
{

    /**
     * Retrieve configuration value
     *
     * @param string $path
     * @return array|string|bool
     */
    public function getConfig($path)
    {
        $config = array(
            'expert' => array(
                'country' => array(
                    'FR' => array(
                        'max_weight' => 30000,
                        'type' => array(
                            'without_signature' => 'COLD',
                            'with_signature'    => 'COL',
                        )
                    ),
                ),
            ),
            'home' => array(
                'country' => array(
                    'FR' => array(
                        'max_weight' => 30000,
                        'type' => array(
                            'without_signature' => 'DOM',
                            'with_signature'    => 'DOS',
                        )
                    ),
                    'BE' => array(
                        'max_weight' => 30000,
                        'type' => array(
                            'without_signature' => 'BOM',
                            'with_signature'    => 'BOS',
                        )
                    )
                ),
            ),
            'pickup' => array(
                'country' => array(
                    'FR' => array(
                        'option_inter' => '0',
                        'max_weight'   => 30000,
                    ),
                    'BE' => array(
                        'option_inter' => '2',
                        'max_weight'   => 20000,
                    ),
                    'NL' => array(
                        'option_inter' => '1',
                        'max_weight'   => 20000,
                    ),
                    'DE' => array(
                        'option_inter' => '1',
                        'max_weight'   => 20000,
                    ),
                    'GB' => array(
                        'option_inter' => '1',
                        'max_weight'   => 20000,
                    ),
                    'LU' => array(
                        'option_inter' => '1',
                        'max_weight'   => 20000,
                    ),
                    'ES' => array(
                        'option_inter' => '1',
                        'max_weight'   => 20000,
                    ),
                ),
            ),
            'international' => array(
                'country' => '::_internationalCountries',
            ),
            'domtom' => array(
                'country' => '::_domtomCountries',
            ),
            'country_settings' => array(
                'FR' => array(
                    'phone_regex'   => '/^0(6|7)[0-9]{8}$/',
                    'phone_example' => '0610203040',
                    'phone_code'    => '',
                ),
                'BE' => array(
                    'phone_regex'   => '/^\+324[0-9]{8}$/',
                    'phone_example' => '+32 410203040',
                    'phone_code'    => '+32',
                ),
                'NL' => array(
                    'phone_regex'   => '/^\+316[0-9]{8}$/',
                    'phone_example' => '+31 610203040',
                    'phone_code'    => '+31',
                ),
                'DE' => array(
                    'phone_regex'   => '/^\+491[5-7]{1}[0-9]{7,8}$/',
                    'phone_example' => '+49 161020304',
                    'phone_code'    => '+49',
                ),
                'GB' => array(
                    'phone_regex'   => '/^\+447[3-9]{1}[0-9]{8}$/',
                    'phone_example' => '+44 7510203040',
                    'phone_code'    => '+44',
                ),
                'LU' => array(
                    'phone_regex'   => '/^\+3526[0-9]{8}$/',
                    'phone_example' => '+352 621203040',
                    'phone_code'    => '+352',
                ),
                'ES' => array(
                    'phone_regex'   => '/^\+346[0-9]{8}$/',
                    'phone_example' => '+34 610203040',
                    'phone_code'    => '+34',
                ),
            ),
        );

        Mage::dispatchEvent('laposte_colissimo_load_config_before', array('path' => $path, 'config' => $config));

        $keys = explode('/', $path);

        $skip = false;
        foreach ($keys as $i => $key) {
            if ($skip) {
                $skip = false;
                continue;
            }
            if (isset($config[$key])) {
                $config = $config[$key];
                if (is_string($config)) {
                    if (preg_match('/^::/', $config)) {
                        $method = preg_replace('/^::/', '', $config);
                        $config = $this->$method();
                        $skip = true;
                    }
                }
            } else {
                $config = false;
                break;
            }
        }

        Mage::dispatchEvent('laposte_colissimo_load_config_after', array('path' => $path, 'config' => $config));

        return $config;
    }

    /**
     * Retrieve International country
     *
     * @return array
     */
    protected function _internationalCountries()
    {
        return array(
            'max_weight' => 30000,
            'type' => array(
                'with_signature' => 'COLI',
            )
        );
    }

    /**
     * Retrieve Dom-Tom country
     *
     * @return array
     */
    protected function _domtomCountries()
    {
        return array(
            'max_weight' => 30000,
            'type' => array(
                'without_signature' => 'COM',
                'with_signature'    => 'CDS',
            )
        );
    }

    /**
     * Mobile phone number validation
     *
     * @param string $phone
     * @param string $countryId
     * @return bool
     */
    public function isValidMobileNumber($phone, $countryId)
    {
        $regex = $this->getConfig('country_settings/' . $countryId . '/phone_regex');

        $valid = true;

        if ($regex) {
            if (!preg_match($regex, $phone)) {
                $valid = false;
            }
        }

        return $valid;
    }

    /**
     * Retrieve phone number with or without code
     *
     * @param string $telephone
     * @param string $countryId
     * @param string $type
     * @return string
     */
    public function getTelephone($telephone, $countryId, $type)
    {
        $phoneCode = $this->getPhoneCode($countryId);

        if ($phoneCode) {
            if ($type == 'add') {
                $telephone = $phoneCode . $telephone;
            }
            if ($type == 'remove') {
                $telephone = str_replace($phoneCode, '', $telephone);
            }
        }

        return $telephone;
    }

    /**
     * Retrieve phone code
     *
     * @param string $countryId
     * @return string
     */
    public function getPhoneCode($countryId)
    {
        return $this->getConfig('country_settings/' . $countryId . '/phone_code');
    }

    /**
     * Retrieve country
     *
     * @param string $countryId
     * @param string $postcode
     * @return string
     */
    public function getCountry($countryId, $postcode = null)
    {
        if ($countryId == 'MC') { // Monaco
            $countryId = 'FR';
        }

        if ($postcode) {
            if ($countryId == 'FR') {
                $countryId = $this->getDomTomCountry($postcode);
            }
        }

        return $countryId;
    }

    /**
     * Retrieve Dom Tom Country with postcode
     *
     * @param string $postcode
     * @return string
     */
    public function getDomTomCountry($postcode)
    {
        $countryId = 'FR';
        $postcodes = $this->getDomTomPostcodes();

        $postcode = preg_replace('/\s+/', '', $postcode);
        foreach ($postcodes as $code => $regex) {
            if (preg_match($regex, $postcode)) {
                $countryId = $code;
                break;
            }
        }
        return $countryId;
    }

    /**
     * Retrieve Dom-Tom countries code ISO-2
     *
     * @return array
     */
    public function getDomTomCountries()
    {
        return array(
            'GP', // Guadeloupe
            'MQ', // Martinique
            'GF', // Guyane
            'RE', // La réunion
            'PM', // St-Pierre-et-Miquelon
            'YT', // Mayotte
            'TF', // Terres-Australes
            'WF', // Wallis-et-Futuna
            'PF', // Polynésie Française
            'NC', // Nouvelle-Calédonie
        );
    }

    /**
     * Retrieve Dom-Tom postcodes
     *
     * @return array
     */
    public function getDomTomPostcodes()
    {
        return array(
            'GP' => '/^971[0-9]{2}$/', // Guadeloupe
            'MQ' => '/^972[0-9]{2}$/', // Martinique
            'GF' => '/^973[0-9]{2}$/', // Guyane
            'RE' => '/^974[0-9]{2}$/', // La réunion
            'PM' => '/^975[0-9]{2}$/', // St-Pierre-et-Miquelon
            'YT' => '/^976[0-9]{2}$/', // Mayotte
            'TF' => '/^984[0-9]{2}$/', // Terres-Australes
            'WF' => '/^986[0-9]{2}$/', // Wallis-et-Futuna
            'PF' => '/^987[0-9]{2}$/', // Polynésie Française
            'NC' => '/^988[0-9]{2}$/', // Nouvelle-Calédonie
        );
    }

    /**
     * Retrieve API configuration
     *
     * @return array
     */
    public function getApiConfig()
    {
        return array(
            'wsdl'     => 'https://ws.colissimo.fr/pointretrait-ws-cxf/PointRetraitServiceWS/2.0?wsdl',
            'login'    => Mage::getStoreConfig('carriers/pickup/account_number'),
            'password' => Mage::getStoreConfig('carriers/pickup/account_password'),
        );
    }

    /**
     * Retrieve log file
     *
     * @return string
     */
    public function getLogFile()
    {
        return 'colissimo.log';
    }

    /**
     * Convert Weight with rate
     *
     * @param float $weight
     * @return int
     */
    public function convertWeight($weight)
    {
        $rate = Mage::getStoreConfig('shipping/colissimo/weight_rate') ?: 1;

        return $weight * $rate;
    }

    /**
     * Retrieve skin anme
     *
     * @return string
     */
    public function getSkinName()
    {
        return Mage::getStoreConfig('carriers/pickup/skin');
    }

    /**
     * Retrieve skin data
     *
     * @return array
     */
    public function getSkinData()
    {
        $skin = Mage::getConfig()->getNode('colissimo/skins/' . $this->getSkinName());

        $data = array();

        if ($skin) {
            $data = $skin->asArray();
        }

        return $data;
    }

    /**
     * Retrieve configured skins
     *
     * @return Mage_Core_Model_Config_Element
     */
    public function getSkins()
    {
        return Mage::getConfig()->getNode('colissimo/skins')->asArray();
    }

    /**
     * Check if we can show map
     *
     * @return int
     */
    public function canShowMap()
    {
        return Mage::getStoreConfigFlag('carriers/pickup/google_show_map') ? 1 : 0;
    }

    /**
     * Retrieve Google Map API key
     *
     * @return string
     */
    public function getMapApiKey()
    {
        return Mage::getStoreConfig('carriers/pickup/google_api_key');
    }

    /**
     * Retrieve if method is active
     *
     * @param $method
     * @return bool
     */
    public function isActive($method)
    {
        return Mage::getStoreConfigFlag('carriers/' . $method . '/active');
    }

    /**
     * Check store is secure
     *
     * @return bool
     */
    public function isSecure()
    {
        return Mage::getStoreConfigFlag('web/secure/use_in_frontend');
    }

    /**
     * Retrieve Shipping Method active countries
     *
     * @param string $shippingMethod
     * @return string
     */
    public function getSpecificCountry($shippingMethod)
    {
        return Mage::getStoreConfig('carriers/' . $shippingMethod . '/specificcountry');
    }

    /**
     * Retrieve Shipping Method signature type
     *
     * @param string $shippingMethod
     * @return string
     */
    public function getSignatureType($shippingMethod)
    {
        return Mage::getStoreConfig('carriers/' . $shippingMethod . '/signature');
    }

    /**
     * Check if Shipping label module from Magentix is enabled
     *
     * @return bool
     */
    public function isShippingLabelEnabled()
    {
        return Mage::helper('core')->isModuleEnabled('LaPoste_Label');
    }

}