<?php
/**
 * Magento
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/osl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@magentocommerce.com so we can send you a copy immediately.
 *
 * @category EDN2
 * @package EDN2_Core
 * @copyright Copyright (c) 2014 Benjamin Denizart www.envoidunet.com
 * @license http://opensource.org/licenses/osl-3.0.php Open Software License (OSL 3.0)
 *
 */
class EDN2_Core_Helper_Api extends Mage_Core_Helper_Abstract {
	/**
	 *
	 * @var EDN2_Core_Model_Api_Soap_Api
	 */
	protected $api;

	public function getApiLogin() {
		return defined('EDN2_API_LOGIN') ? EDN2_API_LOGIN : Mage::getStoreConfig ( 'edn2_api/credentials/login' );
	}
	public function getApiPassword() {
		return defined('EDN2_API_PASSWORD') ? EDN2_API_PASSWORD : Mage::getStoreConfig ( 'edn2_api/credentials/password' );
	}

	public function isProxyEnabled() {
		return (bool) Mage::getStoreConfig ( 'edn2_api/network/proxy_enable' );
	}

	public function getProxyHost() {
		return Mage::getStoreConfig ( 'edn2_api/network/proxy_host' );
	}

	public function getProxyPort() {
		return Mage::getStoreConfig ( 'edn2_api/network/proxy_port' );
	}

	public function getFromCompany() {
		return Mage::getStoreConfig ( 'edn2_api/address/company' );
	}
	public function getFromFirstname() {
		return Mage::getStoreConfig ( 'edn2_api/address/firstname' );
	}
	public function getFromLastname() {
		return Mage::getStoreConfig ( 'edn2_api/address/lastname' );
	}
	public function getFromStreetLine1() {
		return Mage::getStoreConfig ( 'edn2_api/address/street1' );
	}
	public function getFromStreetLine2() {
		return Mage::getStoreConfig ( 'edn2_api/address/street2' );
	}
	public function getFromPostcode() {
		return Mage::getStoreConfig ( 'edn2_api/address/postcode' );
	}
	public function getFromCity() {
		return Mage::getStoreConfig ( 'edn2_api/address/city' );
	}
	public function getFromCountry() {
		return Mage::getStoreConfig ( 'edn2_api/address/country' );
	}
	public function getFromTelephone() {
		return Mage::getStoreConfig ( 'edn2_api/address/telephone' );
	}
	public function getFromEmail() {
		return Mage::getStoreConfig ( 'edn2_api/address/email' );
	}
	public function getApi() {
		if (is_null ( $this->api )) {
			$this->api = Mage::getSingleton ( 'edn2core/api_soap_api' );
			$this->api->setLogin ( $this->getApiLogin (), $this->getApiPassword () );
		}
		return $this->api;
	}

	public function getCarriers($service = null)
	{
        $result = null;

		try {
			$response = $this->getApi ()->getCarriers();

			$response = json_decode(json_encode($response));

			if ($response) {
				if (is_array ( $response->error )) {
					$response->error = current ( $response->error );
				}

				if ($response->error->error_id != 0) {
					throw new Exception ( Mage::helper ( 'edn2core' )->__ ( 'Api Error: %s %s', $response->error->error_message, $response->error->error_description ) );
				}

				if ($response->response && is_object ( $response->response )) {
					if (is_object ( $response->response->carriers )) {
						$response->response->carriers = array (
							$response->response->carriers
						);
					}

					$result = json_decode(json_encode($response->response->carriers), true);
				}
			}
		} catch ( Exception $e ) {
			throw $e;
		}

		if(empty($result)) {
			throw new Exception(Mage::helper ( 'edn2core' )->__ ( 'Unexpected result from getCarriers Edn Api method' ));
		}

		$result = array_filter($result, function ($carrier) {
			return !in_array($carrier['carrier'], array(
				'express', 'international', 'relay', 'standard', 'internationaleco'
			));
		});

		if($service) {
			$result = array_filter($result, function($carrier) use($service) {
				return $carrier['service'] === $service;
			});
		}

		uasort($result, function ($a, $b) {
			return strcasecmp($a['carrier_name'], $b['carrier_name']);
		});

		return array_values($result);
	}

	/**
	 * @param $carrierKey
	 * @return array|null
	 */
	public function getCarrierByKey($carrierKey)
	{
		$carriers = $this->getCarriers();

		foreach($carriers as $carrier) {
			if($carrier['carrier'] === $carrierKey)
				return $carrier;
		}

		return null;
	}

	public function getQuoteSimple($toCompany, $toPostcode, $toCity, $toCountry, $weight) {
		try {
			$request = array ();
			$request ['weight'] = $weight;
			$request ['from'] = array ();
			$request ['from'] ['company'] = $this->getFromCompany();
			$request ['from'] ['postcode'] = $this->getFromPostcode ();
			$request ['from'] ['city'] = $this->getFromCity ();
			$request ['from'] ['country'] = $this->getFromCountry ();
			$request ['to'] = array ();
            $request ['to'] ['company'] = $toCompany;
			$request ['to'] ['postcode'] = $toPostcode;
			$request ['to'] ['city'] = $toCity;
			$request ['to'] ['country'] = $toCountry;
			$request ['packages'] = array ();
			$request ['packages'] [] = array (
					'quantity' => 1,
					'weight' => $weight
			);
			$response = $this->getApi ()->getQuote ( $request );

			$response = json_decode(json_encode($response));

			$quotes = array ();
			if ($response) {
				if (is_array ( $response->error )) {
					$response->error = current ( $response->error );
				}

				if ($response->error->error_id != 0) {
					throw new Exception ( Mage::helper ( 'edn2core' )->__ ( 'Api Error: %s %s', $response->error->error_message, $response->error->error_description ) );
				}

				if ($response->response && is_object ( $response->response )) {
					if (is_object ( $response->response->quotes )) {
						$response->response->quotes = array (
								 $response->response->quotes
						);
					}

					foreach ( $response->response->quotes as $quote ) {
						$quotes [$quote->carrier] = $quote->price;
					}
				}
			}
			return $quotes;
		} catch ( Exception $e ) {
			throw $e;
		}
	}


	public function getQuoteFromShippingRateRequest(Mage_Shipping_Model_Rate_Request $shippingRateRequest) {
		try {
			$request = array ();
			$request ['weight'] = $shippingRateRequest->getPackageWeight ();
			$request ['from'] = array ();
            $request ['from'] ['company'] = $this->getFromCompany();
			$request ['from'] ['postcode'] = $this->getFromPostcode ();
			$request ['from'] ['city'] = $this->getFromCity ();
			$request ['from'] ['country'] = $this->getFromCountry ();
			$request ['to'] = array ();
			$request ['to'] ['postcode'] = $shippingRateRequest->getDestPostcode ();
			$request ['to'] ['city'] = $shippingRateRequest->getDestCity ();
			$request ['to'] ['country'] = $shippingRateRequest->getDestCountryId ();
			$request ['packages'] = array ();
			$request ['packages'] [] = array (
					'quantity' => 1,
					'weight' => $shippingRateRequest->getPackageWeight ()
			);

			$response = $this->getApi ()->getQuote ( $request );

			$response = json_decode(json_encode($response));

			$quotes = array ();
			if ($response) {
				if (is_array ( $response->error )) {
					$response->error = current ( $response->error );
				}

				if ($response->error->error_id != 0) {
					throw new Exception ( Mage::helper ( 'edn2core' )->__ ( 'Api Error: %s %s', $response->error->error_message, $response->error->error_description ) );
				}

				if ($response->response && is_object ( $response->response )) {
					if (is_object ( $response->response->quotes )) {
						$response->response->quotes = array (
								 $response->response->quotes
						);
					}

					foreach ( $response->response->quotes as $quote ) {
						$quotes [$quote->carrier] = $quote->price;
					}
				}
			}
			return $quotes;
		} catch ( Exception $e ) {
			throw $e;
		}
	}
	public function getQuoteFromSalesQuote(Mage_Sales_Model_Quote $quote) {
		try {
			$request = array ();
			$request ['weight'] = $quote->getShippingAddress ()->getWeight ();
			$request ['from'] = array ();
            $request ['from'] ['company'] = $this->getFromCompany();
			$request ['from'] ['postcode'] = $this->getFromPostcode ();
			$request ['from'] ['city'] = $this->getFromCity ();
			$request ['from'] ['country'] = $this->getFromCountry ();
			$request ['to'] = array ();
            $request ['to'] ['company'] = $quote->getShippingAddress ()->getCompany();
			$request ['to'] ['postcode'] = $quote->getShippingAddress ()->getPostcode ();
			$request ['to'] ['city'] = $quote->getShippingAddress ()->getCity ();
			$request ['to'] ['country'] = $quote->getShippingAddress ()->getCountry ();
			$request ['packages'] = array ();
			$request ['packages'] [] = array (
					'quantity' => 1,
					'weight' => $quote->getShippingAddress ()->getWeight ()
			);
			$response = $this->getApi ()->getQuote ( $request );

			$response = json_decode(json_encode($response));

			$quotes = array ();
			if ($response) {
				if (is_array ( $response->error )) {
					$response->error = current ( $response->error );
				}

				if ($response->error->error_id != 0) {
					throw new Exception ( Mage::helper ( 'edn2core' )->__ ( 'Api Error: %s %s', $response->error->error_message, $response->error->error_description ) );
				}

				if ($response->response && is_object ( $response->response )) {
					if (is_object ( $response->response->quotes )) {
						$response->response->quotes = array (
								 $response->response->quotes
						);
					}

					foreach ( $response->response->quotes as $quote ) {
						$quotes [$quote->carrier] = $quote->price;
					}
				}
			}
			return $quotes;
		} catch ( Exception $e ) {
			throw $e;
		}
	}
	public function createShipment(Mage_Sales_Model_Order $order, $carrier, $relayid = null, $weight = array(), $format = 'A6') {
		try {
			$request = array ();
			$request ['reference'] = $order->getIncrementId ();
			$request ['value'] = $order->getGrandTotal ();
			$request ['carrier'] = $carrier;

			$carrierDesc = Mage::helper('edn2core/api')->getCarrierByKey($carrier);

			if($carrierDesc['service'] === 'relay')
				$request ['relay_id'] = ($relayid) ? $relayid : $order->getEdn2Relayid();

			$request ['format'] = $format;
			$request ['from'] = array ();
			$request ['from'] ['company'] = $this->getFromCompany ();
			$request ['from'] ['firstname'] = $this->getFromFirstname ();
			$request ['from'] ['lastname'] = $this->getFromLastname ();
			$request ['from'] ['address1'] = $this->getFromStreetLine1 ();
			$request ['from'] ['address2'] = $this->getFromStreetLine2 ();
			$request ['from'] ['postcode'] = $this->getFromPostcode ();
			$request ['from'] ['city'] = $this->getFromCity ();
			$request ['from'] ['country'] = $this->getFromCountry ();
			$request ['from'] ['email'] = $this->getFromEmail ();
			$request ['from'] ['phone'] = $this->getFromTelephone ();
			$request ['to'] ['company'] = $order->getShippingAddress ()->getCompany ();
			$request ['to'] ['firstname'] = $order->getShippingAddress ()->getFirstname ();
			$request ['to'] ['lastname'] = $order->getShippingAddress ()->getLastname ();
			$request ['to'] ['address1'] = $order->getShippingAddress ()->getStreet1 ();
			$request ['to'] ['address2'] = $order->getShippingAddress ()->getStreet2 ();
			$request ['to'] ['postcode'] = $order->getShippingAddress ()->getPostcode ();
			$request ['to'] ['city'] = $order->getShippingAddress ()->getCity ();
			$request ['to'] ['state'] = $order->getShippingAddress ()->getRegion ();
			$request ['to'] ['country'] = $order->getShippingAddress ()->getCountry ();
			$request ['to'] ['email'] = $order->getCustomerEmail ();
			$request ['to'] ['phone'] = $order->getShippingAddress ()->getTelephone ();
			$request ['to'] ['mobilephone'] = $order->getShippingAddress ()->getMobile();
			$request ['packages'] = array ();
			$products = array ();

			/* @var $item Mage_Sales_Model_Order_Item */
			foreach ( $order->getAllItems () as $item ) {

				if ($item->getProductType () != Mage_Catalog_Model_Product_Type::TYPE_SIMPLE)
					continue;

				if (($item->getQtyInvoiced () - $item->getQtyRefunded ()) <= 0)
					continue;

				$products [] = array (
						'name' => $item->getName (),
						'country_orig' => 'FR',
						'price' => ($item->getQtyInvoiced () > 0) ? ($item->getRowTotal () / $item->getQtyInvoiced ()) : $item->getRowTotal (),
						'qty' => ( int ) $item->getQtyInvoiced (),
						'weight' => $item->getWeight ()
				);
			}

			$request ['packages'] [] = array (
					'quantity' => 1,
					'weight' => reset($weight),
					'products' => $products
			);

			for($i = 1; $i < count($weight); $i++) {
				$request ['packages'] [] = array (
						'quantity' => 1,
						'weight' => $weight[$i],
						'products' => $products
				);
			}

			$response = $this->getApi ()->createShipment ( $request );

			$response = json_decode(json_encode($response));

			if ($response) {
				if (is_array ( $response->error )) {
					$response->error = current ( $response->error );
				}

				if ($response->error->error_id != 0) {
					$translatedMessage = Mage::helper ( 'edn2core' )->__ ( $response->error->error_message );
					$translatedDescription = Mage::helper ( 'edn2core' )->__ ( $response->error->error_description );
					throw new Exception ( Mage::helper ( 'edn2core' )->__ ( 'Order: %s, Api Error: %s %s (%s)', $order->getIncrementId (), $translatedMessage, $translatedDescription,  $response->error->error_id ), $response->error->error_id );
				}

				if($response->response) {
					if (is_object ( $response->response->shipments )) {
						$response ->response->shipments = array (
								$response->response->shipments
						);
					}

					$trackingNumber = array();

					foreach($response->response->shipments as $shipment) {
						$trackingNumber[] = $shipment->trackingnumber;
					}


					return array (
							'price' => $response->response->price,
							'pdf' => $response->response->pdf,
							'trackingnumber' => $trackingNumber
					);
				}
			}
			throw new Exception ( Mage::helper ( 'edn2core' )->__ ('Internal Error'));
		} catch ( Exception $e ) {
			throw $e;
		}
	}

	public function findRelays($carrier, $country, $postcode, $city) {
		try {
			$request = array ();
			$request['carrier'] = $carrier;
			$request['country'] = $country;
			$request['postcode'] = $postcode;
			$request['city'] = $city;

			$response = $this->getApi()->findRelays($request);

			$response = json_decode(json_encode($response));

			if ($response) {
				if (is_array ( $response->error )) {
					$response->error = current ( $response->error );
				}

				if ($response->error->error_id != 0) {
					$translatedMessage = Mage::helper ( 'edn2core' )->__ ( $response->error->error_message );
					$translatedDescription = Mage::helper ( 'edn2core' )->__ ( $response->error->error_description );
					throw new Exception ( Mage::helper ( 'edn2core' )->__ ( 'FindRelays | Api Error: %s %s (%s)', $translatedMessage, $translatedDescription,  $response->error->error_id ), $response->error->error_id );
				}

				if($response->response) {
					if (is_object ( $response->response->relays )) {
						$response ->response->relays = array (
								$response->response->relays
						);
					}

					$relays = array();
					$coords = array();

					$i= 1;
					foreach($response->response->relays as $relaySoap) {
					    $coords[] = array(floatval(str_replace(",", ".", $relaySoap->latitude)), floatval(str_replace(",", ".", $relaySoap->longitude)));
						$pinIdx = str_pad($i, 2, '0', STR_PAD_LEFT);
						$relay = array (
							"id" => trim($relaySoap->relay_id),
							"street1" => trim($relaySoap->name),
							"street2" => "",
							"street3" => trim($relaySoap->address1),
							"street4" => (!empty($relaySoap->address2)) ? trim($relaySoap->address2) : '',
							"postcode" => trim($relaySoap->postcode),
							"city" => trim($relaySoap->city),
							"country" => trim($relaySoap->country),
							"tips" => '',
							"lat" => floatval(str_replace(",", ".", $relaySoap->latitude)),
							"lon" => floatval(str_replace(",", ".", $relaySoap->longitude)),
							"monday" => array(),
							"tuesday" => array(),
							"wednesday" => array(),
							"thursday" =>array(),
							"friday" => array(),
							"saturday" => array(),
							"sunday" => array(),
							"picture" => (!empty($relaySoap->picture)) ? trim($relaySoap->picture) : '' ,
							"thumbnail" => str_replace("http:", "", Mage::getDesign()->getSkinUrl("images/edn2/{$carrier}/pin_edn_{$pinIdx}.png"))
						);


						if(is_object($relaySoap->openingHours))
							$relaySoap->openingHours = array($relaySoap->openingHours);

						foreach($relaySoap->openingHours as $openingHour) {
							if($openingHour->hours) {
								$openingHour->hours = explode(' ', $openingHour->hours);
							}
							switch((int) $openingHour->day) {
								case 1:
									$relay['monday'] = $openingHour->hours;
									break;
								case 2:
									$relay['tuesday'] = $openingHour->hours;
									break;
								case 3:
									$relay['wednesday'] = $openingHour->hours;
									break;
								case 4:
									$relay['thursday'] = $openingHour->hours;
									break;
								case 5:
									$relay['friday'] = $openingHour->hours;
									break;
								case 6:
									$relay['saturday'] = $openingHour->hours;
									break;
								case 7:
									$relay['sunday'] = $openingHour->hours;
									break;
							}
						}

						$relays[] = $relay;
						$i++;
					}

					return array (
					        'pin' => str_replace("http:", "", Mage::getDesign()->getSkinUrl("images/edn2/{$carrier}/pin_edn.png")),
					        'center' => Mage::helper('edn2core/math')->getCenterFromDegrees($coords),
					        'carrier' => $carrier,
							'relays' => $relays
					);
				}
			}
		} catch(Exception $e) {
			throw $e;
		}
	}

	public function getRelay($carrier, $relayId, $country) {
		try {
			$request = array ();
			$request['carrier'] = $carrier;
			$request['relay_id'] = $relayId;
			$request['country'] = $country;

			$response = $this->getApi()->getRelay($request);

			$response = json_decode(json_encode($response));

			if ($response) {
				if (is_array ( $response->error )) {
					$response->error = current ( $response->error );
				}

				if ($response->error->error_id != 0) {
					$translatedMessage = Mage::helper ( 'edn2core' )->__ ( $response->error->error_message );
					$translatedDescription = Mage::helper ( 'edn2core' )->__ ( $response->error->error_description );
					throw new Exception ( Mage::helper ( 'edn2core' )->__ ( 'FindRelays | Api Error: %s %s (%s)', $translatedMessage, $translatedDescription,  $response->error->error_id ), $response->error->error_id );
				}

				if($response->response) {


						$relaySoap = $response->response->relay;
						$relay = array(
								"id" => trim($relaySoap->relay_id),
								"street1" => trim($relaySoap->name),
								"street2" => "",
								"street3" => trim($relaySoap->address1),
								"street4" => trim($relaySoap->address2),
								"postcode" => trim($relaySoap->postcode),
								"city" => trim($relaySoap->city),
								"country" => trim($relaySoap->country),
								"tips" => '',
								"lat" => floatval(str_replace(",", ".", $relaySoap->latitude)),
								"lon" => floatval(str_replace(",", ".", $relaySoap->longitude)),
								"monday" => array(),
								"tuesday" => array(),
								"wednesday" => array(),
								"thursday" =>array(),
								"friday" => array(),
								"saturday" => array(),
								"sunday" => array(),
								"picture" => trim($relaySoap->picture),
								"thumbnail" => "//www.mondialrelay.fr/images/gmappins/gmap_pin1.png"
						);


						if(is_object($relaySoap->openingHours))
							$relaySoap->openingHours = array($relaySoap->openingHours);

						foreach($relaySoap->openingHours as $openingHour) {
							if($openingHour->hours) {
								$openingHour->hours = explode(' ', $openingHour->hours);
							}
							switch((int) $openingHour->day) {
								case 1:
									$relay['monday'] = $openingHour->hours;
									break;
								case 2:
									$relay['tuesday'] = $openingHour->hours;
									break;
								case 3:
									$relay['wednesday'] = $openingHour->hours;
									break;
								case 4:
									$relay['thursday'] = $openingHour->hours;
									break;
								case 5:
									$relay['friday'] = $openingHour->hours;
									break;
								case 6:
									$relay['saturday'] = $openingHour->hours;
									break;
								case 7:
									$relay['sunday'] = $openingHour->hours;
									break;
							}

						}


					return $relay;
				}
			}
		} catch(Exception $e) {
			throw $e;
		}
	}
}
