<?php

/**
 * Magento
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/osl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@magentocommerce.com so we can send you a copy immediately.
 *
 * @category EDN2
 * @package EDN2_Core
 * @copyright Copyright (c) 2014 Benjamin Denizart www.envoidunet.com
 * @license http://opensource.org/licenses/osl-3.0.php Open Software License (OSL 3.0)
 *
 */
class EDN2_Core_Model_Api_Soap_Api
{
    const WSDL = 'http://api.envoidunet.com/?wsdl';

    /**
     *
     * @var Zend_Soap_Client
     */
    protected $soap = null;

    /**
     *
     * @var string
     */
    protected $apiLogin = null;

    /**
     *
     * @var string
     */
    protected $apiPassword = null;

    /**
     * @var array
     */
    protected static $cache = [];

    /**
     */
    public function __construct()
    {
        $this->soap = new Zend_Soap_Client (self::WSDL);
        if ($this->getApiHelper()->isProxyEnabled()) {
            $this->soap->setProxyHost($this->getApiHelper()->getProxyHost());
            $this->soap->setProxyPort($this->getApiHelper()->getProxyPort());
        }
    }

    /**
     * @param string $apiLogin
     * @param string $apiPassword
     */
    public function setLogin($apiLogin, $apiPassword)
    {
        $this->apiLogin = $apiLogin;
        $this->apiPassword = $apiPassword;
    }

    /**
     * @return object
     */
    public function getCarriers()
    {
        return $this->_call(__FUNCTION__, array(), true);
    }

    /**
     * @param array $request
     * @return object
     */
    public function getQuote(array $request)
    {
        return $this->_call(__FUNCTION__, $request, true);
    }

    /**
     *
     * @param array $request
     * @return object
     */
    public function createShipment(array $request)
    {
        return $this->_call(__FUNCTION__, $request, false);
    }

    /**
     * @param array
     * @return object
     */
    public function findRelays(array $request)
    {
        return $this->_call(__FUNCTION__, $request, true);
    }

    /**
     * @param array $request
     * @return object
     */
    public function getRelay(array $request)
    {
        return $this->_call(__FUNCTION__, $request, true);
    }

    private function _call($method, array $request, $cache)
    {
        $cacheId = null;

        if ($cache) {
            $cacheId = $method . ':' . md5(json_encode($request));

            if (isset(self::$cache[$cacheId])) {
                $this->getCoreHelper()->debug(__METHOD__ . ' => ' . $cacheId . ' fetched from static cache');
                return self::$cache[$cacheId];
            }
        }

        try {
            $response = $this->soap->$method(array(
                "api_account" => $this->apiLogin,
                "api_key" => $this->apiPassword,
                "origin" => "magento"
            ), $request);
            $this->_logRequest($method);
        } catch (Exception $e) {
            $this->_logRequest($method);
            $this->_logException($method, $e);
            throw $e;
        }

        if ($cache) {
            self::$cache[$cacheId] = $response;
            $this->getCoreHelper()->debug(__METHOD__ . ' => ' . $cacheId . ' saved to static cache');
        }

        return $response;
    }

    /**
     * @param string $method
     * @param Exception $e
     */
    private function _logException($method, Exception $e)
    {
        $this->getCoreHelper()->debug("===== Exception: " . $method);
        $this->getCoreHelper()->debug("======= Message:");
        $this->getCoreHelper()->debug($e->getMessage());
        $this->getCoreHelper()->debug("======= Trace:");
        $this->getCoreHelper()->debug($e->getTraceAsString());
        $this->getCoreHelper()->debug("=============================");
    }

    /**
     * @param string $method
     */
    private function _logRequest($method)
    {
        $this->getCoreHelper()->debug("===== " . $method);
        $this->getCoreHelper()->debug("======= Request:");
        $this->getCoreHelper()->debug($this->soap->getLastRequest());
        $this->getCoreHelper()->debug("======= Response:");
        $this->getCoreHelper()->debug($this->soap->getLastResponse());
        $this->getCoreHelper()->debug("=============================");
    }

    /**
     */
    public function __destruct()
    {
        $this->soap = null;
    }

    /**
     *
     * @return EDN2_Core_Helper_Data
     */
    public function getCoreHelper()
    {
        return Mage::helper('edn2core');
    }

    /**
     *
     * @return EDN2_Core_Helper_Api
     */
    public function getApiHelper()
    {
        return Mage::helper('edn2core/api');
    }
}
