<?php

class EDN2_Logistic_LogisticController_BadRequestException extends \Exception
{
}

class EDN2_Logistic_LogisticController_NotFoundException extends \Exception
{
}

class EDN2_Logistic_LogisticController_BadConfigurationException extends \Exception
{
}

class EDN2_Logistic_LogisticController_UnexpectedException extends \Exception
{
}

/**
 * Class EDN2_Logistic_LogisticController
 */
class EDN2_Logistic_LogisticController extends Mage_Core_Controller_Front_Action
{
    const LOG_REPROCESS = 'log_reprocess';
    const LOG_RECEIVED = 'log_received';
    const LOG_ERROR = 'log_error';
    const LOG_STOCK_ERROR = 'log_stock_error';
    const LOG_LABEL_ERROR = 'log_label_error';
    const LOG_PICKING_STARTED = 'log_picking_started';
    const LOG_SHIPPED = 'log_shipped';
    const LOG_CANCELED = 'log_canceled';

    /**
     * @var array
     */
    private static $_statusesConf = [
        self::LOG_REPROCESS => [
            'state' => 'processing',
            'label' => '[LOG] Retraitement logistique',
            'isCustomerNotified' => false,
            'isVisibleOnFront' => false
        ],
        self::LOG_RECEIVED => [
            'state' => 'processing',
            'label' => '[LOG] Bien reçu',
            'isCustomerNotified' => false,
            'isVisibleOnFront' => false
        ],
        self::LOG_ERROR => [
            'state' => null,
            'label' => '[LOG] Erreur de réception des données',
            'isCustomerNotified' => false,
            'isVisibleOnFront' => false
        ],
        self::LOG_STOCK_ERROR => [
            'state' => null,
            'label' => '[LOG] Pas assez de stock',
            'isCustomerNotified' => false,
            'isVisibleOnFront' => false
        ],
        self::LOG_LABEL_ERROR => [
            'state' => null,
            'label' => '[LOG] Erreur de génération d\'étiquette',
            'isCustomerNotified' => false,
            'isVisibleOnFront' => false
        ],
        self::LOG_PICKING_STARTED => [
            'state' => null,
            'label' => '[LOG] Début de prise en charge',
            'isCustomerNotified' => false,
            'isVisibleOnFront' => false
        ],
        self::LOG_SHIPPED => [
            'state' => 'processing',
            'label' => '[LOG] Expédiée',
            'isCustomerNotified' => false,
            'isVisibleOnFront' => false,
            'autoCompleteStatus' => 'complete'
        ],
        self::LOG_CANCELED => [
            'state' => 'canceled',
            'label' => '[LOG] Annulée',
            'isCustomerNotified' => false,
            'isVisibleOnFront' => false
        ]
    ];

    public function getCatalogProductsAction()
    {
        return $this->_wrapResponse('application/json', false, false, function () {
            /* @var $products Mage_Catalog_Model_Resource_Product_Collection */
            $products = Mage::getResourceModel('catalog/product_collection');
            $products
                ->addAttributeToSelect(array('sku', 'name', 'weight', 'image', 'price'), 'left')
                ->addAttributeToFilter('type_id', Mage_Catalog_Model_Product_Type::TYPE_SIMPLE);

            foreach (['ean', 'hscode', 'country_origin'] as $attrCode) {
                /** @var Mage_Catalog_Model_Resource_Eav_Attribute $attr */
                $attr = Mage::getResourceModel('catalog/eav_attribute')
                    ->loadByCode('catalog_product', $attrCode);

                if ($attr->getId()) {
                    $products
                        ->addAttributeToSelect($attrCode, 'left');
                }
            }

            $products->getSelect()->limit(10000);

            $results = [];

            $itemCallback = function ($args) use (&$results) {
                /* @var $product Mage_Catalog_Model_Product */
                $product = Mage::getModel('catalog/product');
                $product->setData($args['row']);

                try {
                    /* @var $hImage Mage_Catalog_Helper_Image */
                    $hImage = Mage::helper('catalog/image');
                    $url = $hImage->init($product, 'image')->resize(494, 494);
                } catch (Exception $e) {
                    $url = null;
                }

                $results[] = array(
                    "sku" => $product->getSku(),
                    "name" => $product->getName(),
                    "ean" => $product->getEan(),
                    "hscode" => $product->getHscode(),
                    "country_origin" => $product->getCountryOrigin(),
                    "weight" => (float)$product->getWeight(),
                    "price" => (float)$product->getPrice(),
                    "image_url" => (string)$url
                );
            };

            Mage::getSingleton('core/resource_iterator')->walk(
                $products->getSelect(),
                array($itemCallback)
            );

            return json_encode($results);
        });
    }

    public function getSalesOrdersAction()
    {
        return $this->_wrapResponse('application/json', false, false, function () {
            /** @var EDN2_Core_Helper_Api $edn2coreApi */
            $edn2coreApi = Mage::helper('edn2core/api');

            $shipperAddress = [
                'company' => $edn2coreApi->getFromCompany(),
                'firstname' => $edn2coreApi->getFromFirstname(),
                'lastname' => $edn2coreApi->getFromLastname(),
                'street1' => $edn2coreApi->getFromStreetLine1(),
                'street2' => $edn2coreApi->getFromStreetLine2(),
                'postcode' => $edn2coreApi->getFromPostcode(),
                'city' => $edn2coreApi->getFromCity(),
                'country' => $edn2coreApi->getFromCountry(),
                'email' => $edn2coreApi->getFromEmail(),
                'telephone' => $edn2coreApi->getFromTelephone()
            ];

            /* @var $collection Mage_Sales_Model_Resource_Order_Collection */
            $collection = Mage::getResourceModel('sales/order_collection');

            $forceLogCarrier = $this->getRequest()->getQuery('forceLogCarrier');

            if($references = $this->getRequest()->getQuery('references')) {
                $collection->addAttributeToFilter('increment_id', array('in' => $references));
            } else {
                $collection->addAttributeToFilter('state', array('in' => array('processing')));
                $collection->addAttributeToFilter('status', array('in' => array('processing', self::LOG_REPROCESS)));
            }
            $collection->load();

            $orders = array();
            /* @var $salesOrder Mage_Sales_Model_Order */
            foreach ($collection as $salesOrder) {
                $shippingCarrier = $salesOrder->getShippingCarrier();

                if(!$forceLogCarrier) {
                    if (!$shippingCarrier instanceof EDN2_Shipping_Model_Carrier_Abstract) {
                        continue;
                    } else if (!$shippingCarrier->getLogCarrier()) {
                        continue;
                    }
                    $logCarrier = $shippingCarrier->getLogCarrier();
                } else {
                    $logCarrier = $forceLogCarrier;
                }

                $order = array();
                $order['reference'] = $salesOrder->getIncrementId();
                $order['currency'] = $salesOrder->getOrderCurrencyCode();
                $order['grand_total'] = $salesOrder->getGrandTotal();
                $order['date'] = $salesOrder->getCreatedAtDate()->toString(\Zend_Date::ISO_8601);

                $order['recipient_address'] = array();
                $order['shipper_address'] = array();

                $salesOrderShippingAddress = $salesOrder->getShippingAddress();

                $order['carrier_code'] = $logCarrier;

                $order['recipient_address']['company'] = $salesOrderShippingAddress->getCompany();
                $order['recipient_address']['firstname'] = $salesOrderShippingAddress->getFirstname();
                $order['recipient_address']['lastname'] = $salesOrderShippingAddress->getLastname();
                $order['recipient_address']['street1'] = $salesOrderShippingAddress->getStreet1();
                $order['recipient_address']['street2'] = $salesOrderShippingAddress->getStreet2();
                $order['recipient_address']['postcode'] = $salesOrderShippingAddress->getPostcode();
                $order['recipient_address']['city'] = $salesOrderShippingAddress->getCity();
                $order['recipient_address']['state'] = $salesOrderShippingAddress->getRegion();
                $order['recipient_address']['country'] = $salesOrderShippingAddress->getCountry();
                $order['recipient_address']['telephone'] = $salesOrderShippingAddress->getTelephone();
                $order['recipient_address']['email'] = $salesOrderShippingAddress->getEmail();

                if ($salesOrder->getEdn2Relayid()) {
                    $order['recipient_address']['relay_id'] = $salesOrder->getEdn2Relayid();
                }

                $order['shipper_address'] = $shipperAddress;

                $order['items'] = array();

                /* @var $salesOrderItem Mage_Sales_Model_Order_Item */
                foreach ($salesOrder->getItemsCollection() as $salesOrderItem) {
                    if ($salesOrderItem->getProductType() != Mage_Catalog_Model_Product_Type::TYPE_SIMPLE)
                        continue;

                    $sku = $salesOrderItem->getSku();
                    $price = ($salesOrderItem->getPriceInclTax() <= 0 && $salesOrderItem->getParentItemId()) ? $salesOrderItem->getParentItem()->getPriceInclTax() : $salesOrderItem->getPriceInclTax();
                    $rowPrice = ($salesOrderItem->getRowTotalInclTax() <= 0 && $salesOrderItem->getParentItemId()) ? $salesOrderItem->getParentItem()->getRowTotalInclTax() : $salesOrderItem->getRowTotalInclTax();
                    $qty = ($salesOrderItem->getQtyInvoiced() <= 0 && $salesOrderItem->getParentItemId()) ? $salesOrderItem->getParentItem()->getQtyInvoiced() : $salesOrderItem->getQtyInvoiced();

                    $qtyRefunded = ($salesOrderItem->getQtyRefunded() <= 0 && $salesOrderItem->getParentItemId()) ? $salesOrderItem->getParentItem()->getQtyRefunded() : $salesOrderItem->getQtyRefunded();

                    if ($qtyRefunded >= $qty)
                        continue;

                    $qty -= $qtyRefunded;

                    $item = array();

                    if (isset($order['items'][$sku])) {
                        $item = $order['items'][$sku];
                        $item['price'] += $price;
                        $item['row_price'] += $rowPrice;
                        $item['qty'] += $qty;
                    } else {
                        $item['sku'] = $sku;
                        $item['price'] = $price;
                        $item['row_price'] = $rowPrice;
                        $item['qty'] = $qty;
                    }

                    $order['items'][$sku] = $item;
                }

                if (count($order['items']) > 0) {
                    $order['items'] = array_values($order['items']);

                    $orders[] = $order;
                }
            }
            return json_encode($orders);
        });
    }

    public function updateProductStockAction()
    {
        return $this->_wrapResponse('application/json', true, true, function () {
            // RESUPPLY & INVENTORY
            $sku = $this->getRequest()->getQuery('sku', null);
            $qtyFromLog = $this->getRequest()->getQuery('qty', null);

            if ($qtyFromLog === null or !is_numeric($qtyFromLog)) {
                throw new EDN2_Logistic_LogisticController_BadRequestException('Invalid value for qty query parameter');
            }

            $qtyFromLog = floatval($qtyFromLog);

            /* @var $product Mage_Catalog_Model_Product */
            $product = Mage::getModel('catalog/product');
            $productId = $product->getResource()->getIdBySku($sku);

            if (!$productId) {
                throw new \EDN2_Logistic_LogisticController_NotFoundException('Could not find product by sku');
            }

            $product = $product->load($productId);

            if (!$product || !$product->getEntityId()) {
                throw new EDN2_Logistic_LogisticController_UnexpectedException('Error loading product in database');
            }

            if ($product->getTypeId() != 'simple') {
                throw new EDN2_Logistic_LogisticController_BadRequestException('Product to sync is not of type simple');
            }

            /* @var $resource Mage_Core_Model_Resource */
            $resource = Mage::getSingleton('core/resource');
            /* @var $readConnection Magento_Db_Adapter_Pdo_Mysql */
            $readConnection = $resource->getConnection('core_read');

            $qtyOrdered = (float)$readConnection->fetchOne("
                SELECT SUM(qty_ordered)
                FROM sales_flat_order_item sfoi
                INNER JOIN sales_flat_order sfo ON sfoi.order_id = sfo.entity_id
                LEFT JOIN sales_flat_order_payment sfop ON sfoi.order_id = sfop.parent_id
                WHERE sfoi.product_id = {$product->getId()}
                  AND sfo.state NOT IN('closed', 'canceled', 'complete')
                GROUP BY sfoi.product_id
            ");

            $newShopQty = $qtyFromLog - $qtyOrdered;

            if ($newShopQty < 0)
                $newShopQty = 0;

            /* @var $stockItem Mage_CatalogInventory_Model_Stock_Item */
            $stockItem = Mage::getModel('cataloginventory/stock_item')->loadByProduct($product);

            $oldShopQty = $stockItem->getQty();

            $stockItem->setQty($newShopQty);
            $stockItem->setIsInStock((int)($newShopQty > 0));
            $stockItem->setManageStock(1);
            $stockItem->setUseConfigManageStock(1);
            $stockItem->save();

            $parentIds = Mage::getSingleton('catalog/product_type_configurable')->getParentIdsByChild($product->getId());

            if ($newShopQty > 0) {
                foreach ($parentIds as $parentId) {
                    /* @var $stockItem Mage_CatalogInventory_Model_Stock_Item */
                    $stockItem = Mage::getModel('cataloginventory/stock_item')->loadByProduct($parentId);
                    $stockItem->setIsInStock((int)($newShopQty > 0));
                    $stockItem->setManageStock(1);
                    $stockItem->setUseConfigManageStock(1);
                    $stockItem->save();
                }
            } else {
                foreach ($parentIds as $parentId) {
                    $childQties = 0;
                    $childIdsArr = Mage::getSingleton('catalog/product_type_configurable')->getChildrenIds($parentId);
                    foreach ($childIdsArr as $childIds) {
                        foreach ($childIds as $childId) {
                            /* @var $stockItem Mage_CatalogInventory_Model_Stock_Item */
                            $stockItem = Mage::getModel('cataloginventory/stock_item')->loadByProduct($childId);
                            $childQties += $stockItem->getQty();
                        }
                    }

                    if ($childQties <= 0) {
                        /* @var $stockItem Mage_CatalogInventory_Model_Stock_Item */
                        $stockItem = Mage::getModel('cataloginventory/stock_item')->loadByProduct($parentId);
                        $stockItem->setIsInStock((int)($newShopQty > 0));
                        $stockItem->setManageStock(1);
                        $stockItem->setUseConfigManageStock(1);
                        $stockItem->save();
                    }
                }
            }

            return json_encode([
                'productId' => $productId,
                'qtyFromLog' => $qtyFromLog,
                'qtyOrdered' => $qtyOrdered,
                'oldShopQty' => $oldShopQty,
                'newShopQty' => $newShopQty
            ]);
        });
    }

    public function salesOrderReceivedAction()
    {
        return $this->_wrapResponse('application/json', true, false, function () {
            $salesOrder = $this->_getSalesOrderFromRequest();

            $status = self::LOG_RECEIVED;
            $statusData = $this->_getHandledSalesOrderStatusByKey($status);
            $this->_createSalesOrderStatusHistory($salesOrder, $statusData['model']);

            return json_encode('done');
        });
    }

    public function salesOrderErrorAction()
    {
        return $this->_wrapResponse('application/json', true, false, function () {
            $message = $this->getRequest()->getQuery('message', '');
            $salesOrder = $this->_getSalesOrderFromRequest();

            $status = self::LOG_ERROR;

            if (strpos($message, 'Commande en cours de picking') !== false) {
                $status = $salesOrder->getStatus();
            } else if (strpos($message, 'Not enough stock to pick items') !== false) {
                $status = self::LOG_STOCK_ERROR;
            } else if (strpos($message, 'Not enough stock to pick items') !== false) {
                $status = self::LOG_STOCK_ERROR;
            }

            $statusData = $this->_getHandledSalesOrderStatusByKey($status);
            $this->_createSalesOrderStatusHistory($salesOrder, $statusData['model'], '[LOG] Erreur : ' . $message);


            return json_encode('done');
        });
    }

    public function salesOrderPickingStartedAction()
    {
        return $this->_wrapResponse('application/json', true, false, function () {
            $salesOrder = $this->_getSalesOrderFromRequest();
            $statusData = $this->_getHandledSalesOrderStatusByKey(self::LOG_PICKING_STARTED);
            $this->_createSalesOrderStatusHistory($salesOrder, $statusData['model']);

            return json_encode('done');
        });
    }

    public function salesOrderShippedAction()
    {
        return $this->_wrapResponse('application/json', true, true, function () {
            $salesOrder = $this->_getSalesOrderFromRequest();

            if (!($trackings = json_decode($this->getRequest()->getQuery('trackings'), true))) {
                throw new EDN2_Logistic_LogisticController_BadRequestException('Invalid value for trackings query parameter');
            }

            /** @var EDN2_Shipping_Helper_Carrier $carrierHelper */
            $carrierHelper = Mage::helper('edn2shipping/carrier');

            /** @var Mage_Sales_Model_Resource_Order_Shipment_Collection $shipmentsCollection */
            $shipmentsCollection = $salesOrder->getShipmentsCollection();

            /** @var Mage_Sales_Model_Order_Shipment $shipment */
            foreach ($shipmentsCollection as $shipment) {
                $shipment->delete();
            }

            $itemQtys = array();
            /** @var Mage_Sales_Model_Order_Item $orderItem */
            foreach ($salesOrder->getAllItems() as $orderItem) {
                if (($orderItem->getQtyToShip() || $orderItem->getQtyShipped()) && !$orderItem->getIsVirtual()) {
                    $itemQtys[$orderItem->getId()] = $orderItem->getQtyToShip() ? $orderItem->getQtyToShip() : $orderItem->getQtyShipped();
                }
            }

            if (!$itemQtys) {
                throw new EDN2_Logistic_LogisticController_UnexpectedException('Item quantities is empty');
            }

            $firstShipment = null;
            foreach ($trackings as $tracking) {
                if (!$carrier = $carrierHelper->getCarrierInstanceByLogCarrierKey($tracking['carrier_code'])) {
                    continue;
                }

                $shipment = $salesOrder->prepareShipment($itemQtys);
                $shipment->register();

                $shipment->addComment('Expédié par la logistique', false, false);

                /* @var $track Mage_Sales_Model_Order_Shipment_Track */
                $track = Mage::getModel('sales/order_shipment_track');
                $track->setCarrierCode($carrier->getCarrierCode());
                $track->setTitle($carrier->getTitle());
                $track->setNumber($tracking['tracking']);

                $shipment->addTrack($track);

                $shipment->save();

                if(!$firstShipment) {
                    $firstShipment = $shipment;
                }
            }

            if(!$firstShipment) {
                throw new EDN2_Logistic_LogisticController_UnexpectedException('No shipment created');
            }

            $statusData = $this->_getHandledSalesOrderStatusByKey(self::LOG_SHIPPED);
            $this->_createSalesOrderStatusHistory($salesOrder, $statusData['model']);

            if(!empty($statusData['conf']['autoCompleteStatus'])) {
                $this->_setOrderToCompleteStatus($salesOrder, $firstShipment, $statusData['conf']['autoCompleteStatus']);
            }

            return json_encode('done');
        });
    }

    /**
     *
     */
    public function createLogisticSalesOrderStatesAction()
    {
        return $this->_wrapResponse('application/json', true, false, function () {
            $response = [];

            foreach (self::$_statusesConf as $status => $data) {
                $response[$status] = [];
                /** @var Mage_Sales_Model_Order_Status $statusEntity */
                $salesOrderStatus = $this->_getSalesOrderStatusByKey($status);
                if (!$salesOrderStatus->getId()) {
                    $salesOrderStatus = new Mage_Sales_Model_Order_Status();
                    $salesOrderStatus->setData('status', $status);
                    $salesOrderStatus->setData('label', $data['label']);
                    $salesOrderStatus->save();

                    if ($state = $data['state']) {
                        $salesOrderStatus->assignState($state);
                    }

                    $response[$status]['created'] = true;
                }

                $response[$status]['state'] = $salesOrderStatus->getState();
            }

            return json_encode($response);
        });
    }

    public function getInvoicesPdfAction()
    {
        return $this->_wrapResponse('application/pdf', false, true, function () {
            $salesOrder = $this->_getSalesOrderFromRequest();

            /** @var Mage_Sales_Model_Entity_Order_Invoice_Collection $orderInvoiceCollection */
            $orderInvoiceCollection = Mage::getResourceModel('sales/order_invoice_collection');

            $invoices = $orderInvoiceCollection
                ->setOrderFilter($salesOrder->getId())
                ->load();

            if ($invoices->getAllIds()) {
                /** @var Mage_Sales_Model_Order_Pdf_Invoice $pdfInvoice */
                $pdfInvoice = Mage::getModel('sales/order_pdf_invoice');
                $pdf = $pdfInvoice->getPdf($invoices);

                $fileName = 'invoice-' . $salesOrder->getId() . '.pdf';
                $content = $pdf->render();

                $isFile = false;
                $file = null;
                $contentLength = null;
                if (is_array($content)) {
                    if (!isset($content['type']) || !isset($content['value'])) {
                        return $this;
                    }
                    if ($content['type'] == 'filename') {
                        $isFile = true;
                        $file = $content['value'];
                        $contentLength = filesize($file);
                    }
                }

                $this->getResponse()
                    ->setHttpResponseCode(200)
                    ->setHeader('Pragma', 'public', true)
                    ->setHeader('Cache-Control', 'must-revalidate, post-check=0, pre-check=0', true)
                    ->setHeader('Content-type', 'application/pdf', true)
                    ->setHeader('Content-Length', is_null($contentLength) ? strlen($content) : $contentLength)
                    ->setHeader('Content-Disposition', 'attachment; filename="' . $fileName . '"')
                    ->setHeader('Last-Modified', date('r'));

                if (!is_null($content)) {
                    if ($isFile) {
                        $this->getResponse()->clearBody();
                        $this->getResponse()->sendHeaders();

                        $ioAdapter = new Varien_Io_File();
                        if (!$ioAdapter->fileExists($file)) {
                            Mage::throwException(Mage::helper('core')->__('File not found'));
                        }
                        $ioAdapter->open(array('path' => $ioAdapter->dirname($file)));
                        $ioAdapter->streamOpen($file, 'r');
                        while ($buffer = $ioAdapter->streamRead()) {
                            print $buffer;
                        }
                        $ioAdapter->streamClose();
                        if (!empty($content['rm'])) {
                            $ioAdapter->rm($file);
                        }

                        exit(0);
                    } else {
                        return $content;
                    }
                }
            } else {
                throw new EDN2_Logistic_LogisticController_BadRequestException('No invoice for sales order');
            }
        });
    }

    /**
     * @param string $contentType
     * @param bool $requiresTransaction
     * @param bool $emulateAdminArea
     * @param callable $bodyCallback
     * @throws Exception
     */
    private function _wrapResponse($contentType, $requiresTransaction, $emulateAdminArea, callable $bodyCallback)
    {
        $this->getResponse()->setHeader('ContentType', $contentType);

        /* @var $appEmulation Mage_Core_Model_App_Emulation */
        $appEmulation = Mage::getSingleton('core/app_emulation');
        $initialEnvironmentInfo = null;

        if ($emulateAdminArea) {
            $initialEnvironmentInfo = $appEmulation->startEnvironmentEmulation(0, Mage_Core_Model_App_Area::AREA_ADMINHTML);
        }

        /** @var Mage_Core_Model_Mysql4_Abstract $connection */
        $connection = Mage::getSingleton('core/resource')->getConnection('core_write');

        if ($requiresTransaction) {
            $connection->beginTransaction();
        }

        try {
            $response = $bodyCallback();

            if ($requiresTransaction) {
                if ($this->getRequest()->getQuery('dryrun')) {
                    $connection->rollBack();
                } else {
                    $connection->commit();
                }
            }
        } catch (\Exception $e) {
            if ($requiresTransaction) {
                $connection->rollBack();
            }
            $this->getResponse()->setException($e);

            if ($e instanceof EDN2_Logistic_LogisticController_BadRequestException) {
                $this->getResponse()->setHttpResponseCode(400);
            } else if ($e instanceof EDN2_Logistic_LogisticController_NotFoundException) {
                $this->getResponse()->setHttpResponseCode(404);
            } else {
                $this->getResponse()->setHttpResponseCode(500);
            }

            if ($contentType === 'application/json') {
                $response = json_encode([
                    'errMsg' => $e->getMessage(),
                    'errTrace' => $e->getTraceAsString()
                ]);
            } else {
                $response = $e->getMessage();
            }
        }

        if ($emulateAdminArea) {
            $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo);
        }

        if (!is_string($response)) {
            throw new \EDN2_Logistic_LogisticController_UnexpectedException('Response is not a string');
        }

        $this->getResponse()->setBody($response);
    }

    /**
     * @return Mage_Sales_Model_Order
     * @throws EDN2_Logistic_LogisticController_BadRequestException
     * @throws EDN2_Logistic_LogisticController_NotFoundException
     */
    private function _getSalesOrderFromRequest()
    {
        $reference = $this->getRequest()->getQuery('reference');

        if (!$reference) {
            throw new EDN2_Logistic_LogisticController_BadRequestException('Invalid value for reference query parameter');
        }

        /* @var $salesOrder Mage_Sales_Model_Order */
        $salesOrder = Mage::getModel('sales/order')->loadByIncrementId($reference);


        if (!$salesOrder || !$salesOrder->getId()) {
            throw new EDN2_Logistic_LogisticController_NotFoundException('Cannot find any sales order with the given reference');
        }

        return $salesOrder;
    }

    /**
     * @param string $status
     * @return Mage_Sales_Model_Order_Status
     * @throws Exception
     */
    private function _getSalesOrderStatusByKey($status)
    {
        $salesOrderStatusCollection = Mage::getResourceModel('sales/order_status_collection');

        $salesOrderStatusCollection
            ->getSelect()->joinLeft(
                array('state_table' => $salesOrderStatusCollection->getTable('sales/order_status_state')),
                'main_table.status=state_table.status',
                array('state', 'is_default')
            );

        return $salesOrderStatusCollection
            ->addFieldToFilter('main_table.status', $status)
            ->getFirstItem();
    }

    /**
     * @param string $status
     * @return array
     * @throws Exception
     */
    private function _getHandledSalesOrderStatusByKey($status)
    {
        if (!isset(self::$_statusesConf[$status])) {
            throw new \EDN2_Logistic_LogisticController_UnexpectedException('Unknown status');
        }

        if (!($model = $this->_getSalesOrderStatusByKey($status))) {
            throw new \EDN2_Logistic_LogisticController_BadConfigurationException('Cannot find required status ' . $status);
        }

        return [
            'model' => $model,
            'conf' => self::$_statusesConf[$status]
        ];
    }

    /**
     * @param Mage_Sales_Model_Order $salesOrder
     * @param Mage_Sales_Model_Order_Status $salesOrderStatus
     * @param string|null $overrideComment
     * @return Mage_Sales_Model_Order_Status_History
     * @throws EDN2_Logistic_LogisticController_UnexpectedException
     */
    private function _createSalesOrderStatusHistory(Mage_Sales_Model_Order $salesOrder, Mage_Sales_Model_Order_Status $salesOrderStatus, $overrideComment = null)
    {
        if(!$salesOrderStatus) {
            throw new \EDN2_Logistic_LogisticController_UnexpectedException('sales order status is null');
        }

        $statusConf = isset(self::$_statusesConf[$salesOrderStatus->getId()]) ? self::$_statusesConf[$salesOrderStatus->getId()] : null;

        /* @var $salesOrderStatusHistory Mage_Sales_Model_Order_Status_History */
        $salesOrderStatusHistory = Mage::getModel('sales/order_status_history');
        $salesOrderStatusHistory->setOrder($salesOrder);
        $salesOrderStatusHistory->setComment($overrideComment ? $overrideComment : $salesOrderStatus->getData('label'));
        $salesOrderStatusHistory->setIsCustomerNotified($statusConf && !empty($statusConf['isCustomerNotified']));
        $salesOrderStatusHistory->setIsVisibleOnFront($statusConf  && !empty($statusConf['isVisibleOnFront']));
        $salesOrderStatusHistory->setStatus($salesOrderStatus->getData('status'));
        $salesOrder->setData('state', $salesOrderStatus->getState());
        $salesOrder->setStatus($salesOrderStatus->getData('status'));
        $salesOrder->addStatusHistory($salesOrderStatusHistory);

        $salesOrder->save();
        $salesOrderStatusHistory->save();

        return $salesOrderStatusHistory;
    }

    /**
     * @param Mage_Sales_Model_Order $salesOrder
     * @param Mage_Sales_Model_Order_Shipment $shipment
     * @param string $completeStatus
     */
    private function _setOrderToCompleteStatus(Mage_Sales_Model_Order $salesOrder, Mage_Sales_Model_Order_Shipment $shipment, $completeStatus)
    {
        $shipment->sendEmail();
    }
}
