<?php

/**
 * Magento
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/osl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@magentocommerce.com so we can send you a copy immediately.
 *
 * @category EDN2
 * @package EDN2_Shipping
 * @copyright Copyright (c) 2014 Benjamin Denizart www.envoidunet.com
 * @license http://opensource.org/licenses/osl-3.0.php Open Software License (OSL 3.0)
 *
 */
class EDN2_Shipping_Helper_Carrier extends Mage_Core_Helper_Abstract
{
    /**
     * @param string $carrierCode
     * @param string $country
     * @param string $postcode
     * @param string $city
     * @return array
     * @throws Exception
     */
    public function findRelaysByCarrierCode($carrierCode, $country, $postcode, $city)
    {
        if ($carrier = $this->getCarrierInstanceByCarrierCode($carrierCode)) {
            if (!$carrier->isActive()) {
                throw new Exception('Inactive carrier');
            }

            $response = Mage::helper('edn2core/api')->findRelays($carrier->getEdnCarrier(), $country, $postcode, $city);

            if (isset($response['relays']))
                return $response;

        } else {
            throw new Exception('Unable to get carrier instance');
        }

        return array();
    }

    /**
     * @param string $shipmentMethod
     * @param string $country
     * @param string $postcode
     * @param string $city
     * @return array
     * @throws Exception
     */
    public function findRelaysByShippingMethod($shipmentMethod, $country, $postcode, $city)
    {
        if ($carrier = $this->getCarrierInstanceByShippingMethod($shipmentMethod)) {
            if (!$carrier->isActive()) {
                throw new Exception('Inactive carrier');
            }

            $response = Mage::helper('edn2core/api')->findRelays($carrier->getEdnCarrier(), $country, $postcode, $city);

            if (isset($response['relays']))
                return $response;

        } else {
            throw new Exception('Unable to get carrier instance');
        }

        return array();
    }

    /**
     * @param string $carrierKey
     * @return string
     */
    public function getCarrierCodeByEdnCarrierKey($carrierKey)
    {
        return 'edn2' . str_replace('_', '', $carrierKey);
    }

    /**
     * @param string $carrierCode
     * @return EDN2_Shipping_Model_Carrier_Abstract
     */
    public function getCarrierInstanceByCarrierCode($carrierCode)
    {
        $carrier = Mage::getModel('shipping/config')->getCarrierInstance($carrierCode);

        return $carrier instanceof EDN2_Shipping_Model_Carrier_Abstract ? $carrier : null;
    }

    /**
     * @param string $shippingMethod
     * @return EDN2_Shipping_Model_Carrier_Abstract
     */
    public function getCarrierInstanceByShippingMethod($shippingMethod)
    {
        $carrier = $this->getCarrierInstanceByCarrierCode(explode('_', $shippingMethod)[0]);

        return $carrier instanceof EDN2_Shipping_Model_Carrier_Abstract ? $carrier : null;
    }

    /**
     * @param string $carrierKey
     * @return EDN2_Shipping_Model_Carrier_Abstract
     */
    public function getCarrierInstanceByEdnCarrierKey($carrierKey)
    {
        return $this->getCarrierInstanceByCarrierCode($this->getCarrierCodeByEdnCarrierKey($carrierKey));
    }

    /**
     * @param string $logCarrierKey
     * @return EDN2_Shipping_Model_Carrier_Abstract
     */
    public function getCarrierInstanceByLogCarrierKey($logCarrierKey)
    {
        $carriers = array_values(array_filter($this->getCarriers(), function(EDN2_Shipping_Model_Carrier_Abstract $carrier) use ($logCarrierKey) {
            return $carrier->getLogCarrier() === $logCarrierKey;
        }));

        return count($carriers) > 0 ? $carriers[0] : null;
    }

    /**
     * @param string $service
     * @return array
     */
    public function getCarriers($service = null)
    {
        $carriers = array_map(function(array $carrier) {
            return $this->getCarrierInstanceByEdnCarrierKey($carrier['carrier']);
        }, Mage::helper('edn2core/api')->getCarriers($service));

        // filter out carriers that
        return array_values(array_filter($carriers, function($carrier) {
            return $carrier instanceof EDN2_Shipping_Model_Carrier_Abstract;
        }));
    }

    /**
     * @param string $service
     * @return array
     */
    public function getShippingMethods($service = null)
    {
        return array_map(function (EDN2_Shipping_Model_Carrier_Abstract $carrier) {
            return $carrier->getShippingMethod();
        }, $this->getCarriers($service));
    }

    /**
     * @param string $service
     * @return array
     */
    public function getShippingMethodsForJavascript($service = null)
    {
        return array_map(function (EDN2_Shipping_Model_Carrier_Abstract $carrier) {
            return array(
                'shippingMethod' => $carrier->getShippingMethod(),
                'carrierCode' => $carrier->getCarrierCode(),
                'isRelay' => $carrier->isRelay()
            );
        }, $this->getCarriers($service));
    }

    /**
     * @param string $service
     * @return array
     */
    public function getShippingMethodsOptions($service = null)
    {
        $carriers = $this->getCarriers($service);

        $result = array();

        foreach($carriers as $carrier) {
            /* @var EDN2_Shipping_Model_Carrier_Abstract $carrier */
            $result[$carrier->getShippingMethod()] = $carrier->getTitle();
        }

        return $result;
    }

    /**
     * @param string $carrierCode
     * @return bool
     */
    public function isRelayAndActive($carrierCode)
    {
        if ($carrier = $this->getCarrierInstanceByCarrierCode($carrierCode)) {
            return $carrier->isRelay() && $carrier->isActive();
        } else {
            return false;
        }
    }
}