<?php

/**
 * Magento
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/osl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@magentocommerce.com so we can send you a copy immediately.
 *
 * @category EDN2
 * @package EDN2_Shipping
 * @copyright Copyright (c) 2014 Benjamin Denizart www.envoidunet.com
 * @license http://opensource.org/licenses/osl-3.0.php Open Software License (OSL 3.0)
 *
 */
abstract class EDN2_Shipping_Model_Carrier_Abstract extends EDN2_OwebiaShipping2_Model_Carrier_Abstract
{
    protected $_ednCarrier;
    protected $_logCarrier;
    protected $_isRelay;
    protected $_quoteResult;

    public function __construct()
    {
        parent::__construct();

        if (empty($this->_ednCarrier)) {
            throw new Exception(Mage::helper('edn2shipping')->__('protected _ednCarrier property must be set to edn carrier key'));
        }
    }

    public function checkAvailableShipCountries(Mage_Shipping_Model_Rate_Request $request)
    {
        $speCountriesAllow = $this->getConfigData('sallowspecific');
        $showMethod = $this->getConfigData('showmethod');
        $availableCountries = ($this->getConfigData('specificcountry'))
            ? explode(',', $this->getConfigData('specificcountry')) : array();
        $errorMsg = $this->getConfigData('specificerrmsg');

        /*
         * for specific countries, the flag will be 1
         */
        if ($speCountriesAllow && $speCountriesAllow == 1) {
            if ($availableCountries && in_array($request->getDestCountryId(), $availableCountries)) {
                return $this;
            } elseif ($showMethod && (!$availableCountries || ($availableCountries && !in_array($request->getDestCountryId(), $availableCountries)))) {
                $error = Mage::getModel('shipping/rate_result_error');
                $error->setCarrier($this->_code);
                $error->setCarrierTitle($this->getConfigData('title'));
                $error->setErrorMessage($errorMsg ? $errorMsg : Mage::helper('shipping')->__('The shipping module is not available for selected delivery country.'));
                return $error;
            } else {
                /*
                 * The admin set not to show the shipping module if the devliery country is not within specific countries
                 */
                return false;
            }
        }

        return $this;
    }

    /**
     * (non-PHPdoc)
     *
     * @see Mage_Shipping_Model_Carrier_Abstract::collectRates()
     */
    public function collectRates(Mage_Shipping_Model_Rate_Request $request)
    {
        if (!$this->isActive()) {
            return false;
        }

        $price = $this->_getEdnQuote($request, $this->_ednCarrier);

        if (!$price) {
            return false;
        }

        if ($this->isDeclaredWithVAT())
            $price = $price * 1.20;

        /** @var Mage_Shipping_Model_Rate_Result_Method $method */
        $method = Mage::getModel('shipping/rate_result_method');
        $method->setCarrier($this->_code);
        $method->setCarrierTitle($this->getConfigData('title'));
        $method->setMethod($this->_code);
        $method->setMethodTitle($this->getConfigData('name'));
        $method->setMethodDescription($this->getConfigData('description'));
        $method->setPrice($price);
        $method->setCost($price);

        $request->setData('edn_shipping_price', $price);

        $result = parent::collectRates($request);
        if($result && ($cheapestRate = $result->getCheapestRate())) {
            $method->setPrice($cheapestRate->getPrice());
        } else if($this->isOwebiaConfigFilled()) {
            return false;
        }

        if($round_prices = $this->getConfigData('round_prices')) {
            $method->setPrice(ceil($method->getPrice()));
        }

        /** @var Mage_Shipping_Model_Rate_Result $result */
        $result = Mage::getModel('shipping/rate_result');
        $result->append($method);

        Mage::dispatchEvent('edn2shipping_carrier_' . $this->_code . '_collectrates_after', array("object" => $this, "request" => $request, "result" => $result));
        return $result;
    }

    /**
     * @return array
     */
    public function getAllowedMethods()
    {
        return $this->isActive() ? array(
            $this->_code => $this->getName()
        ) : array();
    }

    /**
     * @return string
     */
    public function getDescription()
    {
        return $this->getConfigData('description');
    }

    /**
     * @return string
     */
    public function getEdnCarrier()
    {
        return $this->_ednCarrier;
    }

    /**
     * @return string
     */
    public function getLogCarrier()
    {
        return $this->_logCarrier;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return $this->getConfigData('name');
    }

    /**
     * @param float $price
     * @return float
     */
    public function getPrice($price)
    {
        return Mage::helper('tax')->getShippingPrice($price);
    }

    /**
     *
     */
    public function getShippingMethod()
    {
        return $this->_code . '_' . $this->_code;
    }

    /**
     * @return string
     */
    public function getTitle()
    {
        return $this->getConfigData('title');
    }

    /**
     * @return bool
     */
    public function isActive()
    {
        return parent::isActive() && is_array(Mage::helper('edn2core/api')->getCarrierByKey($this->_ednCarrier));
    }

    /**
     * @return bool
     */
    public function isDeclaredWithVAT()
    {
        return Mage::helper('tax')->shippingPriceIncludesTax();
    }

    /**
     * @return bool
     */
    public function isRelay()
    {
        return $this->_isRelay;
    }

    /**
     * Check if carrier has shipping tracking option available
     *
     * @return boolean
     */
    public function isTrackingAvailable()
    {
        return true;
    }

    /**
     * @param Mage_Shipping_Model_Rate_Request $request
     * @param string $carrier
     * @return bool|mixed
     */
    protected function _getEdnQuote(Mage_Shipping_Model_Rate_Request $request, $carrier)
    {
        if (is_null($this->_quoteResult)) {
            try {
                /** @var EDN2_Core_Helper_Api $api */
                $api = Mage::helper('edn2core/api');
                $this->_quoteResult = $api->getQuoteFromShippingRateRequest($request);
            } catch (Exception $e) {
                return false;
            }
        }
        if (is_array($this->_quoteResult)) {
            foreach ($this->_quoteResult as $key => $price) {
                if ($key == $carrier)
                    return $price;
            }
        }
        return false;
    }
}