<?php
/**
 * Magento
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/osl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@magentocommerce.com so we can send you a copy immediately.
 *
 * @category EDN2
 * @package EDN2_Shipping
 * @copyright Copyright (c) 2014 Benjamin Denizart www.envoidunet.com
 * @license http://opensource.org/licenses/osl-3.0.php Open Software License (OSL 3.0)
 *
 * @method int getTrackingId()
 * @method int getOrderId()
 * @method string getTrackingNumber()
 * @method string getTrackingUrl()
 * @method string getTrackingPdfUrl()
 * @method string getPrice()
 * @method string getShippingMethod()
 * @method string getCreatedAt()
 *
 * @method EDN2_Shipping_Model_Tracking setTrackingId() setTrackingId($value)
 * @method EDN2_Shipping_Model_Tracking setOrderId() setOrderId($value)
 * @method EDN2_Shipping_Model_Tracking setTrackingNumber() setTrackingNumber($value)
 * @method EDN2_Shipping_Model_Tracking setTrackingUrl() setTrackingUrl($value)
 * @method EDN2_Shipping_Model_Tracking setTrackingPdfUrl() setTrackingPdfUrl($value)
 * @method EDN2_Shipping_Model_Tracking setPrice() setPrice($value)
 * @method EDN2_Shipping_Model_Tracking setShippingMethod() setShippingMethod($value)
 * @method EDN2_Shipping_Model_Tracking setCreatedAt() setCreatedAt($value)
 */
class EDN2_Shipping_Model_Tracking extends Mage_Core_Model_Abstract {

	/**
	 * @var Mage_Sales_Model_Order
	 */
	protected $_order;

	protected function _construct() {
		$this->_init('edn2shipping/tracking');
	}
	
	/**
	 * Set created_at parameter
	 *
	 * @return Mage_Core_Model_Abstract
	 */
	protected function _beforeSave() {
		$date = Mage::getModel('core/date')->gmtDate();
		if ($this->isObjectNew() && !$this->getCreatedAt()) {
			$this->setCreatedAt($date);
		}
		return parent::_beforeSave();
	}

	public function setOrder(Mage_Sales_Model_Order $order) {
		$this->_order = $order;
		$this->setOrderId($order->getId());
	}

	public function getOrder() {
		if(is_null($this->_order) && $this->getOrderId()) {
			$this->_order = Mage::getModel('sales/order')->load($this->getOrderId());
		}
		return $this->_order;
	}

	public function generateLabel($shippingMethod = null, $relayid = null, $force = false, $weight = array()) {
		if($this->getTrackingNumber() && $this->getTrackingPdfUrl() && !$force)
			return $this;

		$order = $this->getOrder();
		
		if(!$order)
			throw new \Exception(Mage::helper('edn2shipping')->__('Order not found.'));

		if(!$shippingMethod) {
			$shippingMethod = $order->getShippingMethod();
		} else {
			$order->addStatusHistoryComment('Envoidunet: Force change shipping method from: ' . $order->getShippingMethod() . ' to: ' . $shippingMethod, false);
			$order->setShippingMethod($shippingMethod);
			$shippingMethods = Mage::helper('edn2shipping/carrier')->getShippingMethodsOptions();
			if(isset($shippingMethods[$shippingMethod])) {
				$order->setShippingDescription($shippingMethods[$shippingMethod]);
			}
		}

		$carrierInstance = Mage::helper('edn2shipping/carrier')->getCarrierInstanceByShippingMethod($shippingMethod);

		if($carrierInstance->isRelay()) {
			if(!$relayid) {
				$relayid = $order->getEdn2Relayid();
				if(!$relayid)
					throw new \Exception(Mage::helper('edn2shipping')->__('Relay ID is empty.'));
			} else {
				$order->setEdn2Relayid($relayid);
			}
		}

		if(empty($weight) || !is_array($weight))
			$weight = array($order->getWeight());
		
		$result = Mage::helper('edn2core/api')->createShipment($order, $carrierInstance->getEdnCarrier(), $relayid, $weight);

		$this->setShippingMethod($shippingMethod);
		
		$trackingUrl = array();
		foreach($result['trackingnumber'] as $trackingnumber) {
			$trackingUrl[] = "http://www.envoidunet.com/fr/suivi-colis/" . $trackingnumber;
		}
		
		$trackingUrl = array_unique($trackingUrl);
		
		if($this->getTrackingNumber())
			$result['trackingnumber'] = array_unique(array_merge($result['trackingnumber'], explode('|', $this->getTrackingNumber())));
		
		
		
		$this->setTrackingUrl(implode('|', $trackingUrl));
		$this->setTrackingPdfUrl($result['pdf']);
		$this->setPrice($result['price']);
		$this->setTrackingNumber(implode('|', $result['trackingnumber']));
		
		// Save data if needed
		$order->save();
	}

	public function shipOrder($colispriveJ1 = false) {
		$order = $this->getOrder();
		if(!$order)
			throw new \Exception(Mage::helper('edn2shipping')->__('Order not found.') . " (".$order->getIncrementId().")");

		if(!$order->canShip()) {
			throw new \Exception(Mage::helper('edn2shipping')->__('Cannot do shipment for the order.') . " (".$order->getIncrementId().")");
		}
		
		if($colispriveJ1 && in_array($order->getShippingMethod(), array("edn2francestandard_colisprive"))) {
			if(strtotime(date("Y-m-d 00:00:00.000")) < strtotime($this->getCreatedAt())) {
				$order->addStatusHistoryComment(
						Mage::helper('edn2shipping')->__ ("Colisprivé - Order will be shipped to J+1. (%s)", gmdate('d/m/Y H:i:s e')
				), false);
				$order->save();
				throw new \Exception(Mage::helper('edn2shipping')->__ ( 'Order %s will be shipped tomorrow. (J+1 Enabled)', $order->getRealOrderId () ));
			}
		}
		
		$shipment = $order->prepareShipment();
		$shipment->register();
		
		$order->addStatusHistoryComment('Shipment created through Envoidunet.', false);
		$shipment->addComment('Shipment created through Envoidunet', false, false);

		$title = 'N/A';
		$shippingMethod = $this->getShippingMethod();

		if($carrierInstance = Mage::helper('edn2shipping/carrier')->getCarrierInstanceByShippingMethod($shippingMethod)) {
			$title = $carrierInstance->getTitle();
		}
		
		list($carrierCode, $method) = explode('_', $shippingMethod, 2);
		
		foreach(explode('|', $this->getTrackingNumber()) as $trackingNumber) {
			/* @var $track Mage_Sales_Model_Order_Shipment_Track */
			$track = Mage::getModel('sales/order_shipment_track');
			$track->setCarrierCode($carrierCode);
			$track->setTitle($title);
			$track->setNumber($trackingNumber);
			$shipment->addTrack($track);
		}
		
		$shipment->sendEmail();
		$shipment->save();

		// change order status to complete
		$order->addStatusToHistory(Mage_Sales_Model_Order::STATE_COMPLETE);
		$order->setData('state', Mage_Sales_Model_Order::STATE_COMPLETE);
		$order->save();
	}
}