<?php
/**
 * Copyright © 2016 Magentix. All rights reserved.
 *
 * NOTICE OF LICENSE
 * This source file is subject to commercial licence, do not copy or distribute without authorization
 */

class LaPoste_Colissimo_Model_Observer
{

    const LAPOSTE_COLISSIMO_SESSION_ADDRESS = 'laposte_colissimo_address';

    /**
     * Do not save Address in address book if pickup (customer address save before)
     *
     * @param Varien_Event_Observer $observer
     */
    public function checkAddress(Varien_Event_Observer $observer)
    {
        /* @var $address Mage_Customer_Model_Address */
        $address = $observer->getCustomerAddress();

        if ($address->getColissimoPickupId()) {
            $address->isDeleted(true);
        }

        $address->setColissimoPickupId(null);
        $address->setColissimoNetworkCode(null);
        $address->setColissimoProductCode(null);
    }

    /**
     * Save colissimo shipping data (shipping method save)
     *
     * @param Varien_Event_Observer $observer
     */
    public function setShippingData(Varien_Event_Observer $observer)
    {
        /** @var Mage_Checkout_OnepageController $action */
        $action  = $observer->getControllerAction();
        $request = $action->getRequest();

        $method = $request->getPost('shipping_method');

        /* @var $address Mage_Sales_Model_Order_Address */
        $address = $action->getOnepage()->getQuote()->getShippingAddress();

        if ($address->getColissimoPickupId() && $this->_getAddress()) {
            foreach ($this->_getAddress() as $attribute => $value) {
                $address->setData($attribute, $value);
            }
            $this->_unsAddress();
        }

        $address->setColissimoProductCode(null);
        $address->setColissimoPickupId(null);
        $address->setColissimoNetworkCode(null);

        $address->save();

        if (!$this->_isColissimo($method)) {
            return;
        }

        $data = new Varien_Object();
        $data->setData($request->getPost());
        $data->setAddress($address);

        $alias = 'laposte_colissimo/method';

        if ($method == 'pickup_colissimo') {
            $this->_saveAddress($address, $this->_getSaveAddressData());
            $alias = 'laposte_colissimo/pickup';
        }

        $model = Mage::getModel($alias);
        $result = $model->saveAddress($data);

        if (is_array($result)) {
            $action->getResponse()->setBody(Mage::helper('core')->jsonEncode($result));
        }
    }

    /**
     * Unset colissimo shipping data (shipping and billing address save)
     *
     * @param Varien_Event_Observer $observer
     */
    public function unsetShippingData(Varien_Event_Observer $observer)
    {
        /** @var Mage_Checkout_OnepageController $action */
        $action  = $observer->getControllerAction();

        /* @var $address Mage_Sales_Model_Order_Address */
        $address = $action->getOnepage()->getQuote()->getShippingAddress();

        $address->setColissimoPickupId(null);
        $address->setColissimoNetworkCode(null);
        $address->setColissimoProductCode(null);

        $this->_unsAddress();
    }

    /**
     * Retrieve data to save
     *
     * @return array
     */
    protected function _getSaveAddressData()
    {
        return array(
            'customer_address_id',
            'save_in_address_book',
            'firstname',
            'lastname',
            'company',
            'street',
            'city',
            'postcode',
            'country_id',
            'telephone',
            'fax',
            'region',
            'region_id',
        );
    }

    /**
     * Save current shipping address
     *
     * @param Mage_Sales_Model_Order_Address $address
     * @param array $attributes
     */
    protected function _saveAddress($address, $attributes)
    {
        $save = array();
        foreach ($attributes as $attribute) {
            $save[$attribute] = $address->getData($attribute);
        }

        $this->_getSession()->setData(self::LAPOSTE_COLISSIMO_SESSION_ADDRESS, $save);
    }

    /**
     * Retrieve saved address data
     *
     * @return array
     */
    protected function _getAddress()
    {
        return $this->_getSession()->getData(self::LAPOSTE_COLISSIMO_SESSION_ADDRESS);
    }

    /**
     * Unset saved address data
     */
    protected function _unsAddress()
    {
        $this->_getSession()->unsetData(self::LAPOSTE_COLISSIMO_SESSION_ADDRESS);
    }

    /**
     * Check if method is Colissimo
     *
     * @param $method
     * @return bool
     */
    protected function _isColissimo($method)
    {
        return preg_match('/^(.*)_colissimo$/', $method);
    }

    /**
     * Retrieve checkout session
     *
     * @return Mage_Checkout_Model_Session
     */
    protected function _getSession()
    {
        return Mage::getSingleton('checkout/session');
    }

}